# Repository Guidelines

## Project Structure
- `crates/`
  - `conform` — CLI (Clap commands, I/O, diffs, manifest)
  - `conform_core` — core engine (planner, merge, executor, backup, detect, fsutil, types)
- `sh/` — CI/lint/test/coverage/release scripts
- `www/` — Next.js 15 website (MDX, Tailwind, static export)
- Build output: `__dist/` (configured via `.cargo/config.toml`)

## Toolchains & Runtimes
- Rust: edition 2021; workspace with members `conform` and `conform_core`.
- Node/Website: Next.js 15, React 19, Tailwind 4; Yarn is used (lockfile present). No `engines` pinned.

## User-Facing Overview

### CLI (What, How, Goals)
- What: A Rust CLI that keeps Node.js repos aligned with a template (ESLint/Prettier/Stylelint configs, ignore files, Husky, Tailwind/PostCSS, `package.json` scripts/metadata) using plan → diff → safe apply.
- How:
  - Initialize in a destination repo: `conform init --source <template-dir>` to create `.conform.toml`.
  - Preview changes: `conform plan --config .conform.toml --dest . --diff` (destination‑aware; only includes Next/Tailwind/Git files if detected).
  - Apply changes: `conform apply --config .conform.toml --dest .` (backs up before overwrite/merge; writes a manifest).
  - Inspect quickly: `conform status ...`; rollback: `conform restore --stamp <YYYYMMDD-HHMMSS>`; housekeeping: `conform clean-backups`.
- Goals: Safe, reviewable, minimal changes that reduce config drift without stomping local tweaks; fully local, no telemetry; predictable rollback via backups + manifest; lockfile hygiene.

### Website (What, How, Goals)
- What: Static Next.js site providing product docs, changelog, and versioned download artifacts.
- How:
  - Built with Yarn, MDX content, Tailwind styling; exported statically (`output: 'export'`, `images.unoptimized: true`).
  - Release script publishes binaries to `www/public/release/v<version>/<OS>/...` and updates `www/src/lib/version.js` + `src/app/page.mdx`.
  - Local dev: `cd www && yarn install && yarn dev`; build: `yarn build` (exports to `www/__dist`).
- Goals: Host documentation and releases in a fast, portable static site; integrate seamlessly with the CLI release process; keep presentation consistent and maintenance low.

## Build & Run (CLI)
- Build (debug): `cargo build -p conform`
- Build (release): `cargo build --release -p conform`
- Lint: `sh/ci.sh lint` (rustfmt check + clippy -D warnings)
- Tests: `sh/ci.sh test`
- Coverage (optional): `sh/coverage.sh` (uses `cargo-tarpaulin` if available)
- Release packaging (host only): `sh/release.sh --host-only` (also updates site version and page). Archive formats:
  - Default: zip only
  - Flags: `--zip-only` (default), `--tar-only`, `--both`

## Build & Run (Website)
- Install deps: `cd www && yarn install`
- Dev server: `yarn dev`
- Lint: `yarn lint`
- Build (static export): `yarn build` (runs `next build && next export -o __dist`)
- Output lives under `www/__dist/`; public assets under `www/public/`

## Coding Style & Conventions (Rust)
- Formatting: rustfmt defaults; clippy with no warnings.
- Errors: `anyhow::Result` with `.context(...)` throughout.
- Naming: types/traits `PascalCase`; modules/files/functions/vars `snake_case`.
- JSON: `serde_json` with `preserve_order` for stable plans/diffs/manifests.

## Coding Style & Conventions (Website)
- Formatting: Prettier; linting via ESLint (Next config).
- Components in `www/src/components/` (`PascalCase.jsx`); MDX/route files in `www/src/app/`.
- Styling: Tailwind CSS v4 with PostCSS (`postcss.config.js`); global styles in `www/src/styles/`.
- MDX: integrated via `@next/mdx` with custom `remark/rehype/recma` plugins under `www/mdx/`.

## Testing
- Rust: no tests currently in repo; harness exists (`sh/ci.sh test`). Place unit tests inline or under `crates/*/tests/` for integration.
- Website: no test setup present.

## Release & Artifacts
- `sh/release.sh` builds multi-target binaries, packages `.zip` by default (writes `.zip.sha256`). Tarballs are optional via flags (`--tar-only` or `--both`).
- Artifacts are copied to `www/public/release/v<version>/<OS>/...`.
- Site metadata updated: `www/src/lib/version.js` and the version/date (and changelog snippet) in `www/src/app/page.mdx`.

## Commit & PR Guidelines
- Commit messages must conform to Conventional Commits (commitlint rules).
- Prefer descriptive scope, e.g., `core: adjust package.json merge`, `site: update MDX plugins`.
- PRs: describe change, rationale, and validation steps (commands/diffs). Include screenshots for website changes when helpful. Ensure `sh/ci.sh lint` and `sh/ci.sh test` pass for CLI changes.

## Security & Configuration Notes
- Proprietary license; private repository. No telemetry.
- CLI creates per-file backups `*-conform-<stamp>.backup` and manifests `.conform-manifest-<stamp>.json` in destination projects.

## Proprietary Lifecycle & Collaboration

This project is proprietary, closed-source, and operated without public code hosting or community contributions. Treat all workflows, docs, and automation as private unless explicitly stated otherwise.

- No public hosting (no GitHub/GitLab):
  - Do not add `.github/`, `.gitlab/`, or similar directories, badges, links, or workflows unless explicitly requested for internal mirroring.
  - Avoid adding public-facing badges (build status, downloads, issues) that point to external forges.
  - CI/CD examples may appear in docs as guidance but do not introduce or maintain CI configs in-repo unless asked.

- No external contributors:
  - Do not add or suggest `CONTRIBUTING.md`, `CODE_OF_CONDUCT.md`, issue/PR templates, or “good first issue” labels.
  - Wording in docs must not invite contributions or issue filing; prefer “contact/support” language routed through the website or private channels.
  - “PRs” in guidelines refer to internal change review only.

- Proprietary distribution only:
  - Do not document or facilitate building from source; binaries are distributed via the website only.
  - Do not publish to public registries (e.g., crates.io, npm, PyPI) or official community taps/buckets unless explicitly approved; prefer private or website-hosted artifacts.
  - Keep release automation private (`sh/release.sh`) and focused on website artifacts under `www/public/release/v<version>/...`.

- Documentation lifecycle:
  - README.md is the canonical, public-facing product doc and is published on the website.
  - CHANGELOG.md is maintained per release and prepended into the website; avoid scattering release notes elsewhere.
  - Avoid adding development-internal details (toolchains, local paths, internal infra) to public docs unless they directly help end users.

- Legal & privacy posture:
  - Preserve the proprietary license and remove open-source signaling (SPDX headers, OSS badges) unless specifically required.
  - Maintain “no telemetry” and offline-first behavior; do not add analytics, crash reporting, or network beacons.

## Agent-Specific Instructions

### Rust CLI (conform)
- Behavior parity:
  - Keep Yarn enforcement in `package.json` merges and removal of `package-lock.json`, `npm-shrinkwrap.json`, `pnpm-lock.yaml`.
  - Preserve merge semantics: DeepGeneric for JSON configs; additive `scripts`/metadata; union arrays; `ManualReview` for `next.config.mjs`.
- Style & safety:
  - Rust 2021; rustfmt; clippy with no warnings. Use `anyhow::Result` + `.with_context(...)`.
  - Only write when content changes; keep dry‑run logic; do not change backup naming or manifest schema.
- Planner/detection: Keep current include/exclude globs and Next/Tailwind/Git gating unchanged unless approved.
- CLI surface: Do not rename commands/flags; additions must be backward‑compatible and documented.
- Dev loop: build `cargo build -p conform`; lint `sh/ci.sh lint`; test `sh/ci.sh test`; coverage `sh/coverage.sh`.

### Website (www)
- Package manager: Yarn only (`yarn install`, `yarn dev`, `yarn build`, `yarn lint`). Do not add or commit `package-lock.json`.
- Runtime/stack: Next.js 15, React 19, Tailwind 4, MDX. Keep `pageExtensions: ['js','jsx','mdx']` (no TS pages).
- Static export: Keep `output: 'export'` and `images.unoptimized: true`. Avoid server-only features.
- Layout & naming: Components in `www/src/components/` (`PascalCase.jsx`); routes/MDX in `www/src/app/`; styles in `www/src/styles/`; path alias `@` per `jsconfig.json`.
- Changelog markers in `page.mdx` (`{/* CHANGELOG_START/END */}`) must remain (release script relies on them).
- Lint/format: Run `yarn lint` before PRs; keep Prettier/ESLint configs unchanged.
- Versioning: `sh/release.sh` updates `www/src/lib/version.js` and `page.mdx`; don’t hand‑edit except for verified hotfixes.

### Shared Practices
- Make surgical changes; avoid unrelated refactors. Preserve filenames, APIs, and semantics.
- Do not modify license/EULA, add telemetry, or publish artifacts.
- Do not commit generated content: `__dist/`, backups `*-conform-<stamp>.backup`, manifests `.conform-manifest-*.json`.
- If core logic changes (merge/planner/executor), add targeted Rust tests; for site changes, include screenshots and ensure `yarn build` succeeds.
- Releases: prefer `sh/release.sh --host-only` locally; it updates site versioning and copies artifacts to `www/public/release/v<version>/...`.

## Pre-Commit Checklists

### CLI (Rust)
- Lint: `sh/ci.sh lint` (rustfmt + clippy -D warnings)
- Tests: `sh/ci.sh test` (add/adjust tests for behavior changes)
- Build: `cargo build -p conform` (ensure no warnings/errors)
- Docs: update README/AGENTS.md if flags/behavior changed
- Messages: Conventional Commits format (commitlint)

### Website (www)
- Install (if deps changed): `cd www && yarn install`
- Lint: `yarn lint` (fix Prettier/ESLint issues)
- Build: `yarn build` (static export to `www/__dist/` succeeds)
- Smoke test: `yarn dev` locally for key pages/components
- Assets/MDX: keep `{/* CHANGELOG_START/END */}` markers intact
- Messages: Conventional Commits format (commitlint)
